
#include <windows.h>
#include "resource.h"
#include "gammadlg.h"
#include "main.h"

volatile static HWND ghGammaDialog = NULL;

static HANDLE hGammaDialogThread = NULL;
static DWORD dwGammaDialogTID;
static CRITICAL_SECTION gcsGammaDialog;
static HANDLE hGammaDialogThreadExitedEvent = NULL;

BOOL ShowWinGlideGammaDialog(HWND, float, float, float);
VOID DestroyWinGlideGammaDialog(VOID);

BOOL EnterGammaDialogCriticalSection(VOID);
VOID LeaveGammaDialogCriticalSection(VOID);

volatile struct GammaDialogExchange_t gGammaDialogExchange;

VOID ResetGammaDialogExchangeData(VOID);
BOOL CheckForNewGammaCorrectionValues(float *, float *, float *);

static DWORD WINAPI GammaDialogThreadProc(LPVOID);


#define SLIDER_MIN_100				0 * 100
#define SLIDER_MAX_100				5 * 100
#define SLIDER_LINE_INCREMENT_100	1
#define SLIDER_PAGE_INCREMENT_100	10

static struct GammaDialogStateInfo_t gGammaDialogInfo;


static LRESULT CALLBACK GammaDialog(HWND, UINT, WPARAM, LPARAM);
static VOID UpdateGammaSlider(HWND, DWORD, BOOL, HWND);
static VOID ExchangeUpdatedGammaValues(DWORD, DWORD, DWORD);


BOOL ShowWinGlideGammaDialog(HWND hParentWnd, float redGamma, float greenGamma, float blueGamma) {
	static struct GammaDialogThreadParameters_t threadParams;

	//If the gamma dialog thread does exist
	if (hGammaDialogThread != NULL) {
		HWND hWnd;

		//Enter gamma dialog critical section
		EnterGammaDialogCriticalSection();

		//See if the gamma dialog handle is valid
		//If it is valid, tell the gamma dialog to focus itself
		hWnd = ghGammaDialog;
		if (hWnd != NULL) {
			PostMessage(ghGammaDialog, PWM_FOCUS_GAMMA_DIALOG, 0, 0);
		}

		//Leave gamma dialog critical section
		LeaveGammaDialogCriticalSection();

		return TRUE;
	}

	//If the gamma dialog thread does not exist, create it
	if (hGammaDialogThread == NULL) {
		//Initialize the parameters passed to the thread
		threadParams.hParentWnd = hParentWnd;
		GetWindowRect(hParentWnd, &threadParams.parentRect);

		//Update gamma information before creating the thread
		gGammaDialogInfo.dwRedGamma100 = (DWORD)(redGamma * 100.0f + 0.375f);
		gGammaDialogInfo.dwGreenGamma100 = (DWORD)(greenGamma * 100.0f + 0.375f);
		gGammaDialogInfo.dwBlueGamma100 = (DWORD)(blueGamma * 100.0f + 0.375f);

		//Set to reset automatically and initially be nonsignaled
		hGammaDialogThreadExitedEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
		if (hGammaDialogThreadExitedEvent == NULL) {
			return FALSE;
		}

		//Create the thread
		hGammaDialogThread = CreateThread(NULL, 0, GammaDialogThreadProc, &threadParams, 0, &dwGammaDialogTID);
		if (hGammaDialogThread == NULL) {
			//Free the gamma dialog thread exited event
			CloseHandle(hGammaDialogThreadExitedEvent);
			hGammaDialogThreadExitedEvent = NULL;

			return FALSE;
		}

		//Initialize the critical section object used with this thread
		InitializeCriticalSection(&gcsGammaDialog);
	}

	return TRUE;
}


VOID DestroyWinGlideGammaDialog(VOID) {
	DWORD dwRet;
	BOOL bRet;

	//If the gamma dialog thread does not exist, there is nothing to do
	if (hGammaDialogThread == NULL) {
		return;
	}

	//If the gamma dialog thread has exited, close the handle and do other cleanup
	dwRet = WaitForSingleObject(hGammaDialogThread, 0);
	if (dwRet == WAIT_OBJECT_0) {
		//Close the thread handle
		CloseHandle(hGammaDialogThread);
		hGammaDialogThread = NULL;

		//Delete the critical section object used with this thread
		DeleteCriticalSection(&gcsGammaDialog);

		//Free the gamma dialog thread exited event
		CloseHandle(hGammaDialogThreadExitedEvent);
		hGammaDialogThreadExitedEvent = NULL;

		//Reset the flag indicating that the gamma dialog was destroyed
		gGammaDialogExchange.bDialogWasDestroyed = FALSE;

		return;
	}

	//If the gamma dialog thread has not exited
	
	//Enter the gamma dialog critical section
	EnterGammaDialogCriticalSection();

	//See if the gamma dialog has been destroyed
	bRet = gGammaDialogExchange.bDialogWasDestroyed;

	//If it has not been destroyed, post a close message so it gets destroyed
	if (bRet == FALSE) {
		//Tell the gamma dialog to exit
		PostMessage(ghGammaDialog, PWM_CLOSE_GAMMA_DIALOG, 0, 0);
	}

	//Leave the gamma dialog critical section
	LeaveGammaDialogCriticalSection();

	//Wait for the gamma dialog thread to exit
	if (1) {
		BOOL bRet;
		DWORD dwRet;

		//See if the thread has already exited
		bRet = GetExitCodeThread(hGammaDialogThread, &dwRet);
		//If the thread is still active, wait for it to exit
		if ((bRet == TRUE) && (dwRet == STILL_ACTIVE)) {
			//Wait for the thread to exit
			dwRet = WaitForSingleObject(hGammaDialogThreadExitedEvent, INFINITE);
		}
		//Close the thread handle
		CloseHandle(hGammaDialogThread);
		hGammaDialogThread = NULL;
	}

	//Delete the critical section object used with this thread
	DeleteCriticalSection(&gcsGammaDialog);

	//Free the gamma dialog thread exited event
	CloseHandle(hGammaDialogThreadExitedEvent);
	hGammaDialogThreadExitedEvent = NULL;

	//Reset the flag indicating that the gamma dialog was destroyed
	gGammaDialogExchange.bDialogWasDestroyed = FALSE;

	return;
}

BOOL EnterGammaDialogCriticalSection(VOID) {
	//Exit and return FALSE if the gamma dialog thread does not exist
	if (hGammaDialogThread == NULL) {
		return FALSE;
	}

	//Enter the gamma dialog critical section
	EnterCriticalSection(&gcsGammaDialog);

	//Return TRUE since the gamma dialog thread exists and the critical section was entered
	return TRUE;
}

VOID LeaveGammaDialogCriticalSection(VOID) {
	//Exit if the gamma dialog thread does not exist
	if (hGammaDialogThread == NULL) {
		return;
	}

	//Leave the gamma dialog critical section
	LeaveCriticalSection(&gcsGammaDialog);

	return;
}

VOID ResetGammaDialogExchangeData(VOID) {
	//Reset the gamma dialog exchange structure
	gGammaDialogExchange.bDialogWasDestroyed = FALSE;
	gGammaDialogExchange.bContainsUpdatedGamma = FALSE;
	gGammaDialogExchange.dwRedGamma100 = 100;
	gGammaDialogExchange.dwGreenGamma100 = 100;
	gGammaDialogExchange.dwBlueGamma100 = 100;

	return;
}

BOOL CheckForNewGammaCorrectionValues(float *pRedGamma, float *pGreenGamma, float *pBlueGamma) {
	BOOL bRet;
	BOOL bGammaDialogWasDestroyed;
	BOOL bUpdateGammaTable;
	DWORD dwRedGamma100, dwGreenGamma100, dwBlueGamma100;

	//See if the gamma correction table should be updated
	//Also check if the gamma dialog has been destroyed and do cleanup if this is the case

	//Enter gamma dialog critical section
	bRet = EnterGammaDialogCriticalSection();

	//Nothing to do if the gamma dialog thread does not exist
	if (bRet == TRUE) {
		//See if the gamma dialog was destroyed
		bGammaDialogWasDestroyed = gGammaDialogExchange.bDialogWasDestroyed;
		if (bGammaDialogWasDestroyed == TRUE) {
			//Do not reset the flag indicating that the gamma dialog was destroyed
			//because DestroyGammaDialog must use this flag and takes care of reseting it
		}
		
		//See if the gamma correction table should be updated
		bUpdateGammaTable = gGammaDialogExchange.bContainsUpdatedGamma;
		if (bUpdateGammaTable == TRUE) {
			//Get the new gamma correction values
			dwRedGamma100 = gGammaDialogExchange.dwRedGamma100;
			dwGreenGamma100 = gGammaDialogExchange.dwGreenGamma100;
			dwBlueGamma100 = gGammaDialogExchange.dwBlueGamma100;

			//Reset the flag indicating that updated gamma info is available
			gGammaDialogExchange.bContainsUpdatedGamma = FALSE;
		}

		//Leave gamma dialog critical section
		LeaveGammaDialogCriticalSection();


		//If the gamma dialog was destroyed, wait for it to exit and clean up the thread
		if (bGammaDialogWasDestroyed == TRUE) {
			DestroyWinGlideGammaDialog();
		}


		//Return new gamma correction information if necessary
		if (bUpdateGammaTable == TRUE) {
			//Return the new gamma correction values
			*pRedGamma = (float)dwRedGamma100 / 100.0f;
			*pGreenGamma = (float)dwGreenGamma100 / 100.0f;
			*pBlueGamma = (float)dwBlueGamma100 / 100.0f;

			//Return TRUE since new gamma correction values were returned
			return TRUE;
		}

		//Return FALSE since no new gamma correction values were returned
		return FALSE;
	}

	//Return FALSE since no new gamma correction values were returned
	return FALSE;
}

static DWORD WINAPI GammaDialogThreadProc(LPVOID pParameter) {
	struct GammaDialogThreadParameters_t *pParams;
	MSG msg;
	 
	//Get the pointer to the thread's parameters
	pParams = pParameter;

	//Create the gamma correction dialog
	ghGammaDialog = CreateDialog(ghDllInst, MAKEINTRESOURCE(IDD_GAMMA_CORRECTION), pParams->hParentWnd, GammaDialog);
	if (ghGammaDialog == NULL) {
		//Exit the thread
		goto exit_thread;
	}

	//Show the window
	ShowWindow(ghGammaDialog, SW_SHOW);

    //Main message loop
	while (GetMessage(&msg, NULL, 0, 0)) {
		if (!IsDialogMessage(ghGammaDialog, &msg)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}

exit_thread:
	//Set the event saying that the thread has exited
	SetEvent(hGammaDialogThreadExitedEvent);

	//Exit the thread
	ExitThread(0);
	return 0;
}



static LRESULT CALLBACK GammaDialog(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	WPARAM fCheck;
	WORD wNotifyCode;
	static HWND hRedSlider, hGreenSlider, hBlueSlider;
	static HWND hRedGamma, hGreenGamma, hBlueGamma;
	static HWND hLockSliders;
	SCROLLINFO scrollInfo;
	static DWORD dwRedGamma100, dwGreenGamma100, dwBlueGamma100;
	static DWORD dwOldRedGamma100, dwOldGreenGamma100, dwOldBlueGamma100;
	int nScrollCode;
	HWND hScrollBar;
	static BOOL bLockSliders;
	int iSliderId;
	DWORD dwGamma100;
	int iRet;
	HDC hDC;
	static HBRUSH hRedBrush = NULL, hGreenBrush = NULL, hBlueBrush = NULL;
	HMENU hSystemMenu;

	switch (message) {
        case WM_INITDIALOG:
			//Get initial gamma correction values
			dwRedGamma100 = gGammaDialogInfo.dwRedGamma100;
			dwGreenGamma100 = gGammaDialogInfo.dwGreenGamma100;
			dwBlueGamma100 = gGammaDialogInfo.dwBlueGamma100;

			//Save old gamma correction values in case the Cancel or Reset button is used
			dwOldRedGamma100 = dwRedGamma100;
			dwOldGreenGamma100 = dwGreenGamma100;
			dwOldBlueGamma100 = dwBlueGamma100;

			//Create brushes
			hRedBrush = CreateSolidBrush(RGB(192, 0, 0));
			hGreenBrush = CreateSolidBrush(RGB(0, 192, 0));
			hBlueBrush = CreateSolidBrush(RGB(0, 0, 192));


			//Make sure the gamma correction values are in the right range
			if (dwRedGamma100 < SLIDER_MIN_100) {
				dwRedGamma100 = SLIDER_MIN_100;
			}
			if (dwRedGamma100 > SLIDER_MAX_100) {
				dwRedGamma100 = SLIDER_MAX_100;
			}

			if (dwGreenGamma100 < SLIDER_MIN_100) {
				dwGreenGamma100 = SLIDER_MIN_100;
			}
			if (dwGreenGamma100 > SLIDER_MAX_100) {
				dwGreenGamma100 = SLIDER_MAX_100;
			}

			if (dwBlueGamma100 < SLIDER_MIN_100) {
				dwBlueGamma100 = SLIDER_MIN_100;
			}
			if (dwBlueGamma100 > SLIDER_MAX_100) {
				dwBlueGamma100 = SLIDER_MAX_100;
			}


			//Initialize controls in the dialog box

			//Disable some commands on the system menu
			hSystemMenu = GetSystemMenu(hWnd, FALSE);
			EnableMenuItem(hSystemMenu, SC_SIZE, MF_GRAYED);
			EnableMenuItem(hSystemMenu, SC_MAXIMIZE, MF_GRAYED);

			//Get the handles of some of the controls
			hRedSlider = GetDlgItem(hWnd, IDC_RED_SLIDER);
			hGreenSlider = GetDlgItem(hWnd, IDC_GREEN_SLIDER);
			hBlueSlider = GetDlgItem(hWnd, IDC_BLUE_SLIDER);
			hRedGamma = GetDlgItem(hWnd, IDC_RED_GAMMA);
			hGreenGamma = GetDlgItem(hWnd, IDC_GREEN_GAMMA);
			hBlueGamma = GetDlgItem(hWnd, IDC_BLUE_GAMMA);
			hLockSliders = GetDlgItem(hWnd, IDC_LOCK_SLIDERS);
			
			//The range of the gamma sliders
			scrollInfo.cbSize = sizeof(scrollInfo);
			scrollInfo.fMask = SIF_PAGE | SIF_RANGE;
			scrollInfo.nMin = SLIDER_MIN_100;
			scrollInfo.nMax = SLIDER_MAX_100;
			scrollInfo.nPage = 0;

			SetScrollInfo(hRedSlider, SB_CTL, &scrollInfo, FALSE);			
			SetScrollInfo(hGreenSlider, SB_CTL, &scrollInfo, FALSE);			
			SetScrollInfo(hBlueSlider, SB_CTL, &scrollInfo, FALSE);


			//The gamma sliders and gamma text values
			UpdateGammaSlider(hRedSlider, dwRedGamma100, FALSE, hRedGamma);
			UpdateGammaSlider(hGreenSlider, dwGreenGamma100, FALSE, hGreenGamma);
			UpdateGammaSlider(hBlueSlider, dwBlueGamma100, FALSE, hBlueGamma);


			//The lock sliders check box
			if ((dwRedGamma100 == dwGreenGamma100) && (dwRedGamma100 == dwBlueGamma100)) {
				fCheck = BST_CHECKED;
				bLockSliders = TRUE;
			}
			else {
				fCheck = BST_UNCHECKED;
				bLockSliders = FALSE;
			}
			SendMessage(hLockSliders, BM_SETCHECK, fCheck, 0);


			return TRUE;

		case WM_COMMAND:
			//Get the notification code
			wNotifyCode = HIWORD(wParam);

			switch(LOWORD(wParam)) {
			case IDC_LOCK_SLIDERS:
				//See if the check box was clicked
				if (wNotifyCode == BN_CLICKED) {
					//Update bLockSliders
					iRet = SendMessage(hLockSliders, BM_GETCHECK, 0, 0);
					if (iRet == BST_CHECKED) {
						bLockSliders = TRUE;
					}
					else {
						bLockSliders = FALSE;
					}
				}

				break;

			case IDC_RESET:
				//Reset the gamma correction values
				dwRedGamma100 = dwOldRedGamma100;
				dwGreenGamma100 = dwOldGreenGamma100;
				dwBlueGamma100 = dwOldBlueGamma100;

				//Update all of the sliders
				UpdateGammaSlider(hRedSlider, dwRedGamma100, TRUE, hRedGamma);
				UpdateGammaSlider(hGreenSlider, dwGreenGamma100, TRUE, hGreenGamma);
				UpdateGammaSlider(hBlueSlider, dwBlueGamma100, TRUE, hBlueGamma);

				//Exchange the updated gamma correction values
				ExchangeUpdatedGammaValues(dwRedGamma100, dwGreenGamma100, dwBlueGamma100);

				break;

			case IDOK:
				//Destroy the window
				DestroyWindow(hWnd);

				break;

			case IDCANCEL:
				//See if the new gamma correction values are different than the old ones
				if ((dwRedGamma100 != dwOldRedGamma100) || (dwGreenGamma100 != dwOldGreenGamma100) || (dwBlueGamma100 != dwOldBlueGamma100)) {
					//Exchange the old gamma correction values
					ExchangeUpdatedGammaValues(dwOldRedGamma100, dwOldGreenGamma100, dwOldBlueGamma100);
				}

				//Destroy the window
				DestroyWindow(hWnd);

				break;

			default:
				break;
			}
			break;

		case WM_HSCROLL:
			hScrollBar = (HWND)lParam;

			//Find out which scroll bar the message came from
			if (hScrollBar == hRedSlider) {
				iSliderId = IDC_RED_SLIDER;
				dwGamma100 = dwRedGamma100;
			}
			else if (hScrollBar == hGreenSlider) {
				iSliderId = IDC_GREEN_SLIDER;
				dwGamma100 = dwGreenGamma100;
			}
			else if (hScrollBar == hBlueSlider) {
				iSliderId = IDC_BLUE_SLIDER;
				dwGamma100 = dwBlueGamma100;
			}
			else {
				//Break if the message came from an unknown scroll bar
				break;
			}

			//Get the scroll code
			nScrollCode = (int)LOWORD(wParam);

			switch (nScrollCode) {
			case SB_BOTTOM:
				dwGamma100 = SLIDER_MIN_100;
				break;

			case SB_TOP:
				dwGamma100 = SLIDER_MAX_100;
				break;

			case SB_LINELEFT:
				if (SLIDER_LINE_INCREMENT_100 > dwGamma100) {
					dwGamma100 = 0;
				}
				else {
					dwGamma100 -= SLIDER_LINE_INCREMENT_100;
				}
				break;

			case SB_LINERIGHT:
				dwGamma100 += SLIDER_LINE_INCREMENT_100;
				break;

			case SB_PAGELEFT:
				if (SLIDER_PAGE_INCREMENT_100 > dwGamma100) {
					dwGamma100 = 0;
				}
				else {
					dwGamma100 -= SLIDER_PAGE_INCREMENT_100;
				}
				break;

			case SB_PAGERIGHT:
				dwGamma100 += SLIDER_PAGE_INCREMENT_100;
				break;

			case SB_THUMBTRACK:
				dwGamma100 = HIWORD(wParam);
				break;

			default:
				//Return if the message is not handled
				return TRUE;
				break;
			}

			//Make sure the range of dwGamma100 is valid
			if (dwGamma100 < SLIDER_MIN_100) {
				dwGamma100 = SLIDER_MIN_100;
			}
			if (dwGamma100 > SLIDER_MAX_100) {
				dwGamma100 = SLIDER_MAX_100;
			}

			//See if the sliders are locked
			if (bLockSliders == FALSE) {
				//Update the slider that was moved
				switch (iSliderId) {
				case IDC_RED_SLIDER:
					dwRedGamma100 = dwGamma100;

					UpdateGammaSlider(hRedSlider, dwRedGamma100, TRUE, hRedGamma);

					break;

				case IDC_GREEN_SLIDER:
					dwGreenGamma100 = dwGamma100;

					UpdateGammaSlider(hGreenSlider, dwGreenGamma100, TRUE, hGreenGamma);

					break;

				case IDC_BLUE_SLIDER:
					dwBlueGamma100 = dwGamma100;

					UpdateGammaSlider(hBlueSlider, dwBlueGamma100, TRUE, hBlueGamma);

					break;

				default:
					break;
				}
			}
			else {
				//Update all of the sliders
				dwRedGamma100 = dwGamma100;
				dwGreenGamma100 = dwGamma100;
				dwBlueGamma100 = dwGamma100;

				UpdateGammaSlider(hRedSlider, dwRedGamma100, TRUE, hRedGamma);
				UpdateGammaSlider(hGreenSlider, dwGreenGamma100, TRUE, hGreenGamma);
				UpdateGammaSlider(hBlueSlider, dwBlueGamma100, TRUE, hBlueGamma);
			}

			//Exchange the updated gamma correction values
			ExchangeUpdatedGammaValues(dwRedGamma100, dwGreenGamma100, dwBlueGamma100);

			break;

		case WM_CTLCOLORSCROLLBAR:
			hScrollBar = (HWND)lParam;
			hDC = (HDC)wParam;

			//Find out which scroll bar the message came from
			if (hScrollBar == hRedSlider) {
				if (hRedBrush != NULL) {
					return (LRESULT)hRedBrush;
				}
			}
			else if (hScrollBar == hGreenSlider) {
				if (hGreenBrush != NULL) {
					return (LRESULT)hGreenBrush;
				}
			}
			else if (hScrollBar == hBlueSlider) {
				if (hBlueBrush != NULL) {
					return (LRESULT)hBlueBrush;
				}
			}
			
			return DefWindowProc(hWnd, message, wParam, lParam);

			break;

		case PWM_FOCUS_GAMMA_DIALOG:
			//Restore the window if it is minimized
			if (IsIconic(hWnd)) {
				ShowWindow(hWnd, SW_RESTORE);
			}

			//Focus the window
			SetFocus(hWnd);

			break;

		case PWM_CLOSE_GAMMA_DIALOG:
			//Destroy the window
			DestroyWindow(hWnd);

			break;

		case WM_DESTROY:
			//Delete brushes
			if (hRedBrush != NULL) {
				DeleteObject(hRedBrush);
				hRedBrush = NULL;
			}
			if (hGreenBrush != NULL) {
				DeleteObject(hGreenBrush);
				hGreenBrush = NULL;
			}
			if (hBlueBrush != NULL) {
				DeleteObject(hBlueBrush);
				hBlueBrush = NULL;
			}

			//Enter gamma dialog critical section
			EnterGammaDialogCriticalSection();

			//Set the flag indicating that the gamma dialog has been destroyed
			gGammaDialogExchange.bDialogWasDestroyed = TRUE;

			//Set the dialog box handle to NULL
			ghGammaDialog = NULL;

			//Leave gamma dialog critical section
			LeaveGammaDialogCriticalSection();

			//Post the quit message
			PostQuitMessage(0);

			break;
		
		default:
			return FALSE;
	}
    return TRUE;
}

static VOID UpdateGammaSlider(HWND hSlider, DWORD dwGamma100, BOOL bRedraw, HWND hGamma) {
	SCROLLINFO scrollInfo;
	char szGammaLevel[32];

	//Update the slider
	scrollInfo.cbSize = sizeof(scrollInfo);
	scrollInfo.fMask = SIF_POS;
	scrollInfo.nPos = dwGamma100;
	SetScrollInfo(hSlider, SB_CTL, &scrollInfo, bRedraw);

	//Update the control displaying the gamma value
	wsprintf(szGammaLevel, "%lu.%02lu", dwGamma100 / 100, dwGamma100 % 100);
	SendMessage(hGamma, WM_SETTEXT, 0, (LPARAM)szGammaLevel);

	return;
}

static VOID ExchangeUpdatedGammaValues(DWORD dwRedGamma100, DWORD dwGreenGamma100, DWORD dwBlueGamma100) {
	//Enter the gamma dialog critical section
	EnterCriticalSection(&gcsGammaDialog);

	//Save the new gamma correction values
	gGammaDialogExchange.dwRedGamma100 = dwRedGamma100;
	gGammaDialogExchange.dwGreenGamma100 = dwGreenGamma100;
	gGammaDialogExchange.dwBlueGamma100 = dwBlueGamma100;

	//Set the flag indicating that there are new gamma correction values
	gGammaDialogExchange.bContainsUpdatedGamma = TRUE;

	//Leave the gamma dialog critical section
	LeaveCriticalSection(&gcsGammaDialog);

	return;
}
